package control;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.Map.Entry;

import abstraction.Action;
import abstraction.OnlyReadIterator;
import abstraction.Service;
import abstraction.State;

/**
 * @author Balestra Concetta, Qusa Hani
 * @version 1.2 09/04/05
 * @since JDK 1.6
 */

/**
 * This Class permits to generate the Orchestrator and obtain its result
 *
 */
public class Orchestrator {
	
	private Simulation s;
	private Community community;
	private Service target;
	private Map<OrchestratorKey, Set<Service>> orchestrator;

	public Orchestrator(Community community, Service target) {
		this.community = community;
		this.target = target;
		this.s = new Simulation(target, community);
		s.compositionViaSimulation();
	}
	
	/**
	 * creates the orchestrator generator storing the result in a map with:
	 *   key = a target state, a community state and an action for which they are in simulation
	 *   value = a set of services which are able to perform this simulation between states and for an action indicates in the key
	 *  @return false if orchestrator and simulation set are empty, true otherwise
	 */
	public boolean generateOrchestrator () {
		if (this.s.isEmpty()) 
			return false;
		else {
			this.orchestrator = new HashMap<OrchestratorKey, Set<Service>> ();
			Iterator<SimulatedBy> pairs = s.getSimulationSet();
			while (pairs.hasNext()) {
				SimulatedBy record = pairs.next();
				State ts = record.getTargetState();
				CState cs = record.getCommunityState();
				Iterator<Action> action = this.target.getActions(ts);
				while(action.hasNext()) {
					Action as = action.next();
					Set<Service> services = new HashSet<Service>();
					Iterator<State> nextTS = this.target.getNextStates(ts, as);
					State nextTState = (State)nextTS.next();	
					for (int i = 0; i < cs.getSize(); i++){
						Service serv = this.community.getService(i);
						if (serv.containsPresentAction(cs.get(i), as)) {
							boolean candoit = true;
							Iterator<State> nextState = serv.getNextStates(cs.get(i), as);
							while (nextState.hasNext() && candoit) {
								State next = (State)nextState.next();
								CState hs = cs.buildNewState (i, next );
								SimulatedBy rec = new SimulatedBy (nextTState, hs);
								if (!s.contains(rec))
									candoit = false;		
							}
							if(candoit && !contains(services,serv))
								services.add(serv);
						}
					}
					OrchestratorKey key = new OrchestratorKey (record, as);
					orchestrator.put(key, services);
				}	
			}
		}
		return true;
	}
	
	/**
	 * checks if composition exists, that is if the initial states of target and available services are in the simulation set
	 * @return true if composition exits, otherwise false
	 */
	public boolean checkComposition () {
		SimulatedBy initPair = new SimulatedBy (this.target.getInitialState(), this.community.getInitialStates());
		return this.s.contains(initPair);
	}

	/**
	 * checks if a service is already present in a set of service
	 * @param sts - set of services
	 * @param s - service
	 * @return true if the set contains the specified service
	 */
	private boolean contains(Set<Service> sts, Service s){
		Iterator<Service> it = sts.iterator();
		while (it.hasNext()) {
			Service serv = (Service)it.next();
			if (serv.getName().equals(s.getName()))
				return true;
		}
		return false;
	}
	
	/**
	 * gets an iterator on the element of the map produced by the orchestrator generator 
	 * @return an iterator 
	 */
	public Iterator<Entry<OrchestratorKey, Set<Service>>> getOrchestrator () {
		return new OnlyReadIterator<Entry<OrchestratorKey, Set<Service>>>(this.orchestrator.entrySet().iterator());
	}
	
	/**
	 * gets the set of services which are able to perform a simulation between a target state and a community state with an action 
	 * @param orch - an object OrchestratorKey containing a target state, a community state and an action to check
	 * @return an iterator on a set of services
	 */
	public Iterator<Service> getServicesForStateAction (OrchestratorKey orch) {
		return new OnlyReadIterator<Service>(this.orchestrator.get(orch).iterator());
	}
	
	/**
	 * gets an iterator on the simulation set
	 * @return Iterator on SymulatedBy elements
	 */
	public Iterator<SimulatedBy> getSimulationSet () {
		return this.s.getSimulationSet();
	}
	
	/**
	 * creates a graph representing the Orchestrator Generator obtained from this orchestrator 
	 * @return a service that is the OG
	 */
	public Service createOrchestratorGenerator() {
		if (!checkComposition()) return null;
		StringBuilder s = new StringBuilder("OrchestratorGenerator?transition:"); 
		Set<SimulatedBy> toAdd = new HashSet<SimulatedBy>();  
		Set<SimulatedBy> alreadyAdded = new HashSet<SimulatedBy>();
		Set<SimulatedBy> finalStates = new HashSet<SimulatedBy>();
		
		State initialTS = this.target.getInitialState();
		CState initialCS = this.community.getInitialStates();
		SimulatedBy initialStates = new SimulatedBy(initialTS, initialCS);
		toAdd.add(initialStates);
		while (!toAdd.isEmpty()) {
			Set<SimulatedBy> simulatedPairs = new HashSet<SimulatedBy>();
			simulatedPairs.addAll(toAdd);
			Iterator<SimulatedBy> pair = toAdd.iterator();
			while (pair.hasNext()) {
				SimulatedBy simulated = pair.next();
				State initTS = simulated.getTargetState();
				CState initCS = simulated.getCommunityState();
				alreadyAdded.add(simulated);
				if (this.target.isFinalState(initTS)&& this.community.isFinalCommunityState(initCS)){
					finalStates.add(simulated);
				}
				Iterator<Action> iter = this.target.getActions(initTS);
				while (iter.hasNext()) {
					Action action = (Action)iter.next();
					State nextTS = (State)this.target.getNextStates(initTS, action).next();
					OrchestratorKey orchKey = new OrchestratorKey(simulated, action);
					Iterator<Service> srv = this.getServicesForStateAction(orchKey);
					while (srv.hasNext()) {
						Service serv = srv.next();
						int position = this.community.getServicePosition(serv);
						Iterator<State> next = serv.getNextStates(initCS.get(position), action);
						while (next.hasNext()) {
							State nextState = next.next();
							CState cs = initCS.buildNewState(position, nextState);
							SimulatedBy sim = new SimulatedBy(nextTS, cs);
							if (this.s.contains(sim)) {
								if (!alreadyAdded.contains(sim)) {
									simulatedPairs.add(sim);
								}
								s = s.append(simulated + "-" + action + "," + serv + "-" + sim + ";");
							}
						}
					}
				}
				simulatedPairs.remove(simulated);
				toAdd = simulatedPairs;
			}
		}
		s = s.append("initial:" + initialStates + ";" + "final:");
		Iterator<SimulatedBy> f = finalStates.iterator();
		int count = finalStates.size();
		while (f.hasNext()) {
			SimulatedBy finalSim = f.next(); 
			s = s.append(finalSim);
			count--;
			if (count == 0) 
				s = s.append(".");
			else
				s = s.append(";");
		}
	
		return new Service(s.toString());
	}
	
	public String toString () {
		StringBuilder s = new StringBuilder();
		Iterator<Entry<OrchestratorKey, Set<Service>>> it = orchestrator.entrySet().iterator();
		while (it.hasNext()) {
			Entry<OrchestratorKey, Set<Service>> set =(Entry<OrchestratorKey, Set<Service>>)it.next();
				s = s.append("(" + set.getKey().toString()+ " , "); 
				Set<Service> servs=(Set<Service>)set.getValue();
				Iterator<Service> iter = servs.iterator();
				while (iter.hasNext()) {
					Service serv = (Service)iter.next();
					s = s.append(serv.getName());
				}
				s = s.append(")");
		}
		return s.toString();
	}
}
